// $Id: CString.cpp,v 1.7 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CString.hpp"
using Exponent::Basics::CString;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CString, CCountedObject);

//	===========================================================================
const char CString::CSTRING_EMPTY_STRING[3] = " ";

//	===========================================================================
const char CString::CSTRING_NULL_STRING[] = "\0";

//	===========================================================================
CString::CString(const char *string) : m_string(NULL), m_numberOfCharacters(0)
{
	EXPONENT_CLASS_CONSTRUCTION(CString);
	NULL_POINTER(m_string);
	if (string)
	{
		this->setString(string);
	}
	else
	{
		this->setString(CSTRING_EMPTY_STRING);
	}
}

//	===========================================================================
CString::CString(const CString &string) : m_string(NULL), m_numberOfCharacters(0)
{
	EXPONENT_CLASS_CONSTRUCTION(CString);
	NULL_POINTER(m_string);
	this->setString(string);
}

//	===========================================================================
CString::~CString()
{
	EXPONENT_CLASS_DESTRUCTION(CString);
	FREE_ARRAY_POINTER(m_string);
}

//  ===========================================================================
CString &CString::operator = (const CString &string)
{
	if (this != &string)
	{
		this->setString(string);
	}
	return (*this);
}

//  ===========================================================================
CString &CString::operator = (CString &string)
{
	if (*this != string)
	{
		this->setString(string.getString());
	}
	return (*this);
}

//  ===========================================================================
CString &CString::operator = (const char *string)
{
	this->setString(string);
	return (*this);
}

//  ===========================================================================
CString &CString::operator = (const float value)
{
	char *text = this->toString(value);
	(*this)    = text;
	return (*this);
}

//  ===========================================================================
CString &CString::operator = (const long value)
{
	char *text = this->toString(value);
	(*this)    = text;
	return (*this);
}

//  ===========================================================================
CString &CString::operator = (const int value)
{
	char *text = this->toString(value);
	(*this)    = text;
	return (*this);
}

//  ===========================================================================
CString &CString::operator = (const bool value)
{
	if (value)
	{
		this->setString("True");
	}
	else
	{
		this->setString("False");
	}
	return (*this);
}

//  ===========================================================================
CString &CString::operator += (const CString &string)
{
	this->appendString(string);
	return (*this);
}

//  ===========================================================================
CString &CString::operator += (const char *string)
{
	this->appendString(string);
	return (*this);
}

//  ===========================================================================
bool CString::operator == (const char *equals) const
{
	return (strcmp(m_string, equals) == 0);
}

//  ===========================================================================
bool CString::operator == (const CString &string) const
{
	return (strcmp(m_string, string.getString()) == 0);
}

//  ===========================================================================
bool CString::operator != (const char *string) const
{
	return (!(*this == string));
}

//  ===========================================================================
bool CString::operator != (const CString &string) const
{
	return (!(*this == string));
}

//  ===========================================================================
bool CString::operator ! () const
{
	return (m_numberOfCharacters == 0);
}

//  ===========================================================================
char CString::operator [] (const long index) const
{
	return this->characterAt(index);
}

//  ===========================================================================
double CString::toDouble(const char *text)
{
	return atof(text);
}

//  ===========================================================================
float CString::toFloat(const char *text)
{
	return (float)atof(text);
}

//  ===========================================================================
long CString::toLong(const char *text)
{
	return (long)atol(text);
}

//  ===========================================================================
int CString::toInt(const char *text)
{
	return (int)atoi(text);
}

//  ===========================================================================
char *CString::toString(const float value)
{
	char *text = new char[CSTRING_MIDDLE_TEXT_LENGTH];
	sprintf(text, "%.3f", value);
	return text;
}

//  ===========================================================================
char *CString::toString(const int value)
{
	return toString((long)value);
}

//  ===========================================================================
char *CString::toString(const long value)
{
	char *text = new char[CSTRING_MIDDLE_TEXT_LENGTH];
	sprintf(text, "%.3li", value);
	return text;
}

//  ===========================================================================
void CString::setString(const char *string)
{
	if (string)
	{
		// How many character are we being asked to become?
		long chars = (long)strlen(string);

		// If they want us to grow..
		FREE_ARRAY_POINTER(m_string);
		m_numberOfCharacters = chars + 1;
		m_string = new char[m_numberOfCharacters];

		// Copy the string
		strcpy(m_string, string);
	}
}

//  ===========================================================================
void CString::setString(const CString &string)
{
	this->setString(string.getString());
}

//  ===========================================================================
void CString::setStringWithFormat(const char *text, ...)
{
	// Get a marker for the start
	va_list marker;
	va_start(marker, text);

#ifdef WIN32
	// Store the number of characters
	long numberOfCharacters = _vscprintf(text, marker);
#else
	// No range checking on mac, and i aint about to implement it either!!
	long numberOfCharacters = 1024;	
#endif

	// Create the new string
	char *buffer = new char[numberOfCharacters + 1];

	// Default the buffer
	memset(buffer, 0, numberOfCharacters + 1 * sizeof(char));

#ifdef WIN32
	// Now format the text
	_vsnprintf(buffer, numberOfCharacters, text, marker);
#else
	vsnprintf(buffer, numberOfCharacters, text, marker);
#endif

	// Set the string
	this->setString(buffer);

	// Delete the buffer
	FREE_ARRAY_POINTER(buffer);
}

//  ===========================================================================
void CString::appendString(const char character)
{
	// This is sooooo.... cheating! ;)
	char asString[2];
	asString[0] = character;
	asString[1] = '\0';
	this->appendString(asString);
}

//  ===========================================================================
void CString::appendString(const char *string)
{
	long numChars = (long)strlen(string);

	// Copy the current string
	char *current = new char[m_numberOfCharacters];
	strcpy(current, this->getString());

	FREE_ARRAY_POINTER(m_string);

	// recreate the string at the new length
	m_numberOfCharacters += numChars;
	m_string = new char[m_numberOfCharacters];

	// copy and append
	strcpy(m_string, current);
	strcat(m_string, string);

	// now release the current
	FREE_ARRAY_POINTER(current);
}

//  ===========================================================================
void CString::appendString(const CString &string)
{
	this->appendString(string.getString());
}

//  ===========================================================================
void CString::getString(char *buffer, const long size) const
{
	if (buffer)
	{
		memset(buffer, 0, sizeof(char) * size);
		if (size < m_numberOfCharacters)
		{
			// Buffer is smaller than the string. Need to copy up to the amount in to the buffer
			strncpy(buffer, m_string, size - 1);
			buffer[size - 1] = '\0';
		}
		else
		{
			// Buffer is >= the string. Can do straight copy
			strcpy(buffer, m_string);
		}
	}
}

//  ===========================================================================
void CString::getString(char *buffer) const
{
	this->getString(buffer, CSTRING_SHORT_TEXT_LENGTH);
}

//  ===========================================================================
const char *CString::getString() const
{
	return m_string;
}

//  ===========================================================================
CString CString::getSubString(const long start, const long end) const
{
	// Check the range
	if ((start < end) && ((start >= 0) && (end <= m_numberOfCharacters)))
	{
		// Store the length
		const long length = (end - start) + 2;

		// Create the new buffer
		char *text = new char[length];

		// Copy the text
		memcpy(text, &m_string[start], length * sizeof(char));

		// Store terminator
		text[length - 1] = '\0';

		// Create their string
		CString string(text);

		// Delete the sub string pointer
		FREE_ARRAY_POINTER(text);

		// Return the string
		return string;
	}
	
	// They get an empty string
	return CString(CString::CSTRING_NULL_STRING);
}

//  ===========================================================================
long CString::getNumberOfCharacters() const
{
	return m_numberOfCharacters;
}

//  ===========================================================================
char CString::characterAt(const int index) const
{
	if ((index >= 0) && (index < m_numberOfCharacters))
	{
		return m_string[index];
	}
	return ' ';
}

//  ===========================================================================
void CString::toUpperCase()
{
	for (long i = 0; i < m_numberOfCharacters; i++)
	{
		m_string[i] = toUpperCase(m_string[i]);
	}
}

//  ===========================================================================
void CString::toLowerCase()
{
	for (long i = 0; i < m_numberOfCharacters; i++)
	{
		m_string[i] = toLowerCase(m_string[i]);
	}
}

//  ===========================================================================
void CString::removeTrailingAndLeadingWhiteSpace()
{
	this->removeLeadingWhiteSpace();
	this->removeTrailingWhiteSpace();
}

//  ===========================================================================
void CString::removeLeadingWhiteSpace()
{
	long start = 0;
	while(isWhiteSpace(m_string[start]))
	{
		++start;
	}

	if (start < m_numberOfCharacters)
	{
		this->setString(this->getSubString(start, m_numberOfCharacters));
	}
}

//  ===========================================================================
void CString::removeTrailingWhiteSpace()
{
	long end = m_numberOfCharacters - 2;
	while(end >= 0 && isWhiteSpace(m_string[end]))
	{
		--end;
	}

	if (end >= 0)
	{
		this->setString(this->getSubString(0, end++));
	}
}

//  ===========================================================================
void CString::removeLastCharacter()
{
	if (m_numberOfCharacters > 1)
	{
		char *newString = new char[m_numberOfCharacters];
		memset(newString, 0, m_numberOfCharacters * sizeof(char));
		strcpy(newString, m_string);
		newString[m_numberOfCharacters - 2] = '\0';
		this->setString(newString);
		FREE_ARRAY_POINTER(newString);
	}
}

//  ===========================================================================
void CString::replaceCharWithChar(const char characterToReplace, const char characterToReplaceWith)
{
	for (long i = 0; i < m_numberOfCharacters; i++)
	{
		if (m_string[i] == characterToReplace)
		{
			m_string[i] = characterToReplaceWith;
		}
	}
}

//  ===========================================================================
void CString::replaceCharWithString(const char characterToReplace, const CString &stringToReplaceWith)
{
	// We want to build up a new string
	CString newString = "";

	// Loop through all characters
	for (long i = 0; i < m_numberOfCharacters; i++)
	{
		// Check if its the character
		if (m_string[i] == characterToReplace)
		{
			// First character?
			if (newString == "")
			{
				newString = stringToReplaceWith;
			}
			else
			{
				newString.appendString(stringToReplaceWith);
			}
		}
		else
		{
			// First character
			if (newString == "")
			{
				// Copy character
				char buffer[2];
				buffer[0] = m_string[i];
				buffer[1] = '\0';

				// Assign
				newString = buffer;
			}
			else
			{
				// Append the string
				newString.appendString(m_string[i]);
			}
		}
	}

	// Copy it
	*this = newString;
}

//  ===========================================================================
void CString::replaceStringWithChar(const char characterToReplaceWith, const CString &stringToReplace)
{
	// Find the string
	const long startIndex = this->findSubString(stringToReplace);

	// Check if its in range
	if (startIndex == -1)
	{
		return;
	}

	// Copy the first part of the string
	CString newString = this->getSubString(0, startIndex - 1);

	// Append the character
	newString.appendString(characterToReplaceWith);

	// Now replace the end of the string
	newString.appendString(this->getSubString(startIndex + stringToReplace.getNumberOfCharacters() - 1, this->getNumberOfCharacters() - 1));

	// Copy and we are done
	*this = newString;

	// Now loop inwards to find other versions of the same characters string
	this->replaceStringWithChar(characterToReplaceWith, stringToReplace);
}

//  ===========================================================================
char CString::toUpperCase(const char letter)
{
	return (char)toupper(letter);
}

//  ===========================================================================
char CString::toLowerCase(const char letter)
{
	return (char)tolower(letter);
}

//  ===========================================================================
bool CString::isWhiteSpace(const char letter)
{
	return letter == ' ' || (letter <= 13 && letter >= 9);
}

//  ===========================================================================
bool CString::equals(const char *other) const
{
	return (strcmp(m_string, other) == 0);
}

//  ===========================================================================
bool CString::equals(const CString &string) const
{
	return this->equals(string.getString());
}

//  ===========================================================================
bool CString::equalsIgnoringCase(const char *other) const
{
	return (stricmp(m_string, other) == 0);
}

//  ===========================================================================
bool CString::equalsIgnoringCase(const CString &string) const
{
	return this->equalsIgnoringCase(string.getString());
}

//  ===========================================================================
long CString::findForward(const char character)
{
	long index = -1;
	for (long i = 0; i < m_numberOfCharacters; i++)
	{
		if (m_string[i] == character)
		{
			index = i;
			break;
		}
	}
	return index;
}

//  ===========================================================================
long CString::findBackwards(const char character)
{
	long index = -1;
	for (long i = m_numberOfCharacters - 1; i >= 0; i--)
	{
		if (m_string[i] == character)
		{
			index = i;
			break;
		}
	}
	return index;
}

//  ===========================================================================
long CString::findSubString(const CString &string)
{
	long index = this->findForward(string[0]);

	// Check if the character is found
	if (index == -1)
	{
		return -1;
	}

	// Loop through the characters and find
	if (index + string.getNumberOfCharacters() <= m_numberOfCharacters)
	{
		long startIndex = index;
		for (long i = 0; i < string.getNumberOfCharacters() - 1; i++)
		{
			if (m_string[startIndex++] != string[i])
			{
				return -1;
			}
		}
		return index;
	}

	// Failed
	return -1;
}

#ifndef WIN32
//  ===========================================================================
CFStringRef CString::getStringAsCFStringRef() const
{
	return CFStringCreateWithCString(kCFAllocatorDefault, m_string, kCFStringEncodingMacRoman);
}

//  ===========================================================================
void CString::setStringFromCFStringRef(const CFStringRef &string)
{
	char buffer[1024];
	Boolean b = CFStringGetCString(string, buffer, 1024, kCFStringEncodingMacRoman);
	if (b)
	{
		this->setString(buffer);
	}
}
#endif

//  ===========================================================================
CPascalString CString::getStringAsPascalString()
{
	return CPascalString(m_string);
}

//  ===========================================================================
void CString::getObjectDescription(char *string, const long size) const
{
	this->getString(string, size);
}